//	Altirra - Atari 800/800XL emulator
//	Copyright (C) 2008 Avery Lee
//
//	This program is free software; you can redistribute it and/or modify
//	it under the terms of the GNU General Public License as published by
//	the Free Software Foundation; either version 2 of the License, or
//	(at your option) any later version.
//
//	This program is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//	GNU General Public License for more details.
//
//	You should have received a copy of the GNU General Public License
//	along with this program; if not, write to the Free Software
//	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

#include "stdafx.h"
#include <vd2/system/math.h>
#include <vd2/Riza/display.h>
#include <vd2/Kasumi/pixmap.h>
#include <vd2/Kasumi/pixmapops.h>
#include <vd2/Kasumi/pixmaputils.h>
#include <vd2/Kasumi/triblt.h>
#include "gtia.h"
#include "console.h"
#include <windows.h>

class ATFrameTracker : public vdrefcounted<IVDRefCount> {
public:
	ATFrameTracker() : mActiveFrames(0) { }
	VDAtomicInt mActiveFrames;
};

class ATFrameBuffer : public VDVideoDisplayFrame {
public:
	ATFrameBuffer(ATFrameTracker *tracker) : mpTracker(tracker) {
		++mpTracker->mActiveFrames;
	}

	~ATFrameBuffer() {
		--mpTracker->mActiveFrames;
	}

	const vdrefptr<ATFrameTracker> mpTracker;
	VDPixmapBuffer mBuffer;
};

namespace {
	const uint8 PF0		= 0x01;
	const uint8 PF1		= 0x02;
	const uint8 PF01	= 0x03;
	const uint8 PF2		= 0x04;
	const uint8 PF3		= 0x08;
	const uint8 PF23	= 0x0c;
	const uint8 PF		= 0x0f;
	const uint8 P0		= 0x10;
	const uint8 P1		= 0x20;
	const uint8 P01		= 0x30;
	const uint8 P2		= 0x40;
	const uint8 P3		= 0x80;
	const uint8 P23		= 0xc0;

	static const uint8 kMissileTables[2][16]={
		{ 0, P0, P1, P0|P1, P2, P2|P0, P2|P1, P2|P0|P1, P3, P0|P3, P1|P3, P0|P1|P3, P2|P3, P3|P2|P0, P3|P2|P1, P3|P2|P0|P1 },
		{ 0, PF3, PF3, PF3, PF3, PF3, PF3, PF3, PF3, PF3, PF3, PF3, PF3, PF3, PF3, PF3 }
	};
}

ATGTIAEmulator::ATGTIAEmulator()
	: mpFrameTracker(new ATFrameTracker)
	, mArtifactMode(0)
	, mbShowCassetteIndicator(false)
	, mPreArtifactFrameBuffer(456*262)
{
	mPreArtifactFrame.data = mPreArtifactFrameBuffer.data();
	mPreArtifactFrame.pitch = 456;
	mPreArtifactFrame.palette = mPalette;
	mPreArtifactFrame.data2 = NULL;
	mPreArtifactFrame.data3 = NULL;
	mPreArtifactFrame.pitch2 = 0;
	mPreArtifactFrame.pitch3 = 0;
	mPreArtifactFrame.w = 456;
	mPreArtifactFrame.h = 262;
	mPreArtifactFrame.format = nsVDPixmap::kPixFormat_Pal8;

	mpFrameTracker->AddRef();

	mCONSOL = 7;
	mPRIOR = 0;
	mTRIG[0] = 0x01;
	mTRIG[1] = 0x01;
	mTRIG[2] = 0x01;
	mTRIG[3] = 0x01;
	mAnalysisMode = kAnalyzeNone;
	mStatusFlags = 0;
	mStickyStatusFlags = 0;
	mCassettePos = 0;
	memset(mColorTable, 0, sizeof mColorTable);
	memset(mPlayerCollFlags, 0, sizeof mPlayerCollFlags);
	memset(mMissileCollFlags, 0, sizeof mMissileCollFlags);

	mpPriTable = mPriorityTables[0];
	mpMissileTable = kMissileTables[0];

	mbTurbo = false;

	// init priority tables

	static const int kBasePriorities[4][4]={
		{ P01, P23, PF, PF },
		{ P01, PF, PF, P23 },
		{ PF, PF, P01, P23 },
		{ PF01, P01, P23, PF23 },
	};

	memset(mPriorityTables, 0, sizeof mPriorityTables);
	
	for(int v=0; v<4; ++v) {
		for(int i=0; i<256; ++i) {
			uint8 hibit = 0;
			for(int k=0; k<4; ++k) {
				uint8 bit = kBasePriorities[v][k] & i;
				if (bit) {
					hibit = bit;
					break;
				}
			}

			// "there is no priority between playfields"
			// -- This is critical when mixing priority D3=1 with D2=1;
			//    if PF0 and P5 are active, the split playfield priority
			//    in the D3=1 case does not cause PF0 to activate.
			if (hibit & PF)
				hibit = i & PF;

			uint8 c = 8;
			uint8 d = 8;
			switch(hibit) {
				// playfield hack to make sure PF3 (player 5) always shows up on top -- fixes Last Starfighter meters
				case 1:		c = d = 4;	break;
				case 2:		c = d = 5;	break;
				case 3:		c = d = 5;	break;
				case 4:		c = d = 6;	break;
				case 5:		c = d = 6;	break;
				case 6:		c = d = 6;	break;
				case 7:		c = d = 6;	break;
				case 8:		c = d = 7;	break;
				case 9:		c = d = 7;	break;
				case 10:	c = d = 7;	break;
				case 11:	c = d = 7;	break;
				case 12:	c = d = 7;	break;
				case 13:	c = d = 7;	break;
				case 14:	c = d = 7;	break;
				case 15:	c = d = 7;	break;
				case P0:	c = d = 0;	break;
				case P1:	c = d = 1;	break;
				case P2:	c = d = 2;	break;
				case P3:	c = d = 3;	break;
				case P01:	c = 0; d = 17; break;
				case P23:	c = 2; d = 18; break;
			}

			mPriorityTables[(1<<v)][i] = c;
			mPriorityTables[(1<<v) + 16][i] = d;
		}
	}

	memcpy(mPriorityTables[0], mPriorityTables[1], sizeof mPriorityTables[0]);
	memcpy(mPriorityTables[16], mPriorityTables[17], sizeof mPriorityTables[16]);

	for(int v = 3; v < 31; ++v) {
		uint8 mode = v & 16;
		uint8 pri = v & 15;

		uint8 bit123 = pri & (pri-1);
		uint8 bit0 = pri - bit123;
		const uint8 *tab0 = mPriorityTables[mode + bit0];
		uint8 bit23 = bit123 & (bit123-1);
		uint8 bit1 = bit123 - bit23;
		const uint8 *tab1 = bit1 ? mPriorityTables[mode + bit1] : tab0;
		uint8 bit3 = bit23 & (bit23-1);
		uint8 bit2 = bit23 - bit3;
		const uint8 *tab2 = bit2 ? mPriorityTables[mode + bit2] : tab1;
		const uint8 *tab3 = bit3 ? mPriorityTables[mode + bit3] : tab2;
		
		for(int i=0; i<256; ++i) {
			uint8 c = tab0[i];

			if (c != tab1[i] || c != tab2[i] || c != tab3[i])
				c = 16;

			mPriorityTables[v][i] = c;
		}
	}

	SetPALMode(false);
}

ATGTIAEmulator::~ATGTIAEmulator() {
	if (mpFrameTracker) {
		mpFrameTracker->Release();
		mpFrameTracker = NULL;
	}
}

void ATGTIAEmulator::Init(IATGTIAEmulatorConnections *conn) {
	mpConn = conn;
	mX = 0;
	mY = 0;

	memset(mPlayerPos, 0, sizeof mPlayerPos);
	memset(mMissilePos, 0, sizeof mMissilePos);
	memset(mPlayerSize, 0, sizeof mPlayerSize);
	memset(mPlayerData, 0, sizeof mPlayerData);
	mMissileData = 0;
	memset(mPMColor, 0, sizeof mPMColor);
	memset(mPFColor, 0, sizeof mPFColor);
	RecomputePalette();
}

void ATGTIAEmulator::AdjustColors(double baseDelta, double rangeDelta) {
	mPaletteColorBase += baseDelta;
	mPaletteColorRange += rangeDelta;
	RecomputePalette();
}

void ATGTIAEmulator::SetAnalysisMode(AnalysisMode mode) {
	mAnalysisMode = mode;
}

void ATGTIAEmulator::SetVideoOutput(IVDVideoDisplay *pDisplay) {
	mpDisplay = pDisplay;

	if (!pDisplay) {
		mpFrame = NULL;
		mpDst = NULL;
	}
}

void ATGTIAEmulator::SetCassettePosition(float pos) {
	mCassettePos = pos;
}

void ATGTIAEmulator::SetPALMode(bool enabled) {
	mbPALMode = enabled;

	// These constants, and the algorithm in RecomputePalette(), come from an AtariAge post
	// by Olivier Galibert. Sadly, the PAL palette doesn't quite look right, so it's disabled
	// for now.

	// PAL palette doesn't look quite right.
//	if (enabled) {
//		mPaletteColorBase = -58.0/360.0;
//		mPaletteColorRange = 14.0;
//	} else {
		mPaletteColorBase = -15.0/360.0;
		mPaletteColorRange = 14.4;
//	}

	RecomputePalette();
}

void ATGTIAEmulator::SetConsoleSwitch(uint8 c, bool set) {
	mCONSOL &= ~c;

	if (!set)			// bit is active low
		mCONSOL |= c;
}

void ATGTIAEmulator::DumpStatus() {
	for(int i=0; i<4; ++i) {
		ATConsolePrintf("Player  %d: color = %02x, pos = %02x, size=%d, data = %02x\n"
			, i
			, mPMColor[i]
			, mPlayerPos[i]
			, mPlayerSize[i]
			, mPlayerData[i]
			);
	}

	for(int i=0; i<4; ++i) {
		ATConsolePrintf("Missile %d: color = %02x, pos = %02x, size=%d, data = %02x\n"
			, i
			, mPRIOR & 0x10 ? mPFColor[3] : mPMColor[i]
			, mMissilePos[i]
			, (mMissileSize >> (2*i)) & 3
			, (mMissileData >> (2*i)) & 3
			);
	}

	ATConsolePrintf("Playfield colors: %02x | %02x %02x %02x %02x\n"
		, mPFBAK
		, mPFColor[0]
		, mPFColor[1]
		, mPFColor[2]
		, mPFColor[3]);

	ATConsolePrintf("PRIOR:  %02x (pri=%2d%s%s %s)\n"
		, mPRIOR
		, mPRIOR & 15
		, mPRIOR & 0x10 ? ", pl5" : ""
		, mPRIOR & 0x20 ? ", multicolor" : ""
		, (mPRIOR & 0xc0) == 0x00 ? ", normal"
		: (mPRIOR & 0xc0) == 0x40 ? ", 1 color / 16 lumas"
		: (mPRIOR & 0xc0) == 0x80 ? ", 9 colors"
		: ", 16 colors / 1 luma");

	ATConsolePrintf("VDELAY: %02x\n", mVDELAY);

	ATConsolePrintf("GRACTL: %02x%s%s%s\n"
		, mGRACTL
		, mGRACTL & 0x04 ? ", latched" : ""
		, mGRACTL & 0x02 ? ", player DMA" : ""
		, mGRACTL & 0x01 ? ", missile DMA" : ""
		);

	ATConsolePrintf("CONSOL: %02x%s%s%s%s\n"
		, mCONSOL
		, mCONSOL & 0x08 ? ", speaker" : ""
		, mCONSOL & 0x04 ? ", option" : ""
		, mCONSOL & 0x02 ? ", select" : ""
		, mCONSOL & 0x01 ? ", start" : ""
		);

	uint8 v;
	for(int i=0; i<4; ++i) {
		v = ReadByte(0x00 + i);
		ATConsolePrintf("M%cPF:%s%s%s%s\n"
			, '0' + i
			, v & 0x01 ? " PF0" : ""
			, v & 0x02 ? " PF1" : ""
			, v & 0x04 ? " PF2" : ""
			, v & 0x08 ? " PF3" : "");
	}

	for(int i=0; i<4; ++i) {
		v = ReadByte(0x04 + i);
		ATConsolePrintf("P%cPF:%s%s%s%s\n"
			, '0' + i
			, v & 0x01 ? " PF0" : ""
			, v & 0x02 ? " PF1" : ""
			, v & 0x04 ? " PF2" : ""
			, v & 0x08 ? " PF3" : "");
	}

	for(int i=0; i<4; ++i) {
		v = ReadByte(0x08 + i);
		ATConsolePrintf("M%cPL:%s%s%s%s\n"
			, '0' + i
			, v & 0x10 ? " P0" : ""
			, v & 0x20 ? " P1" : ""
			, v & 0x40 ? " P2" : ""
			, v & 0x80 ? " P3" : "");
	}

	for(int i=0; i<4; ++i) {
		v = ReadByte(0x0c + i);
		ATConsolePrintf("P%cPL:%s%s%s%s\n"
			, '0' + i
			, v & 0x01 ? " PF0" : ""
			, v & 0x02 ? " PF1" : ""
			, v & 0x04 ? " PF2" : ""
			, v & 0x08 ? " PF3" : "");
	}
}

void ATGTIAEmulator::BeginScanline(int y, bool hires) {
	mbHiresMode = hires;

	if (!y && !mpFrame) {
		if (mpDisplay) {
			for(;;) {
				if (mpDisplay->RevokeBuffer(false, ~mpFrame)) {
					VDASSERT(mpFrame);
					break;
				}

				if (mpFrameTracker->mActiveFrames < 3) {
					ATFrameBuffer *fb = new ATFrameBuffer(mpFrameTracker);
					mpFrame = fb;

					fb->mPixmap.format = 0;
					fb->mbAllowConversion = true;
					fb->mbInterlaced = false;
					fb->mFlags = IVDVideoDisplay::kAllFields | IVDVideoDisplay::kVSync;
					break;
				}

				if (mbTurbo) {
					break;
				}

				MSG msg;
				if (WAIT_OBJECT_0 == MsgWaitForMultipleObjects(0, NULL, FALSE, 1, QS_SENDMESSAGE|QS_TIMER)) {
					while(PeekMessage(&msg, NULL, 0, 0, PM_REMOVE)) {
						if (msg.message == WM_QUIT) {
							PostQuitMessage(msg.wParam);
							break;
						}
						TranslateMessage(&msg);
						DispatchMessage(&msg);
					}
				}
			}

			int format = mArtifactMode ? nsVDPixmap::kPixFormat_XRGB8888 : nsVDPixmap::kPixFormat_Pal8;

			if (mpFrame && !mpFrame->mPixmap.format) {
				ATFrameBuffer *fb = static_cast<ATFrameBuffer *>(&*mpFrame);

				if (mArtifactMode)
					fb->mBuffer.init(912, 262, format);
				else
					fb->mBuffer.init(456, 262, format);

				fb->mPixmap = fb->mBuffer;
				fb->mPixmap.palette = mPalette;
			}
		}
	}

	mpDst = NULL;
	
	mY = y;
	mX = 0;
	mLastSyncX = 0;

	if (mpFrame) {
		if (mArtifactMode)
			mpDst = &mPreArtifactFrameBuffer[y * 456];
		else
			mpDst = (uint8 *)mpFrame->mPixmap.data + y * mpFrame->mPixmap.pitch;

		if (y < 262) {
			memset(mpDst, 0, 456);
		} else {
			mpDst = NULL;
		}
	}
	memset(mMergeBuffer, 0, sizeof mMergeBuffer);
	memset(mPlayfieldLumaBytes, 0, sizeof mPlayfieldLumaBytes);
}

void ATGTIAEmulator::EndScanline(uint8 dlControl) {
	if (!mpDst)
		return;

	// obey VBLANK
	if (mY < 8 || mY >= 248) {
		memset(mpDst, mColorTable[8], 456);
	} else {
		Sync();

		mPlayerData[0] = mDelayedPlayerData[0];
		mPlayerData[1] = mDelayedPlayerData[1];
		mPlayerData[2] = mDelayedPlayerData[2];
		mPlayerData[3] = mDelayedPlayerData[3];
		mMissileData = mDelayedMissileData;
	}

	switch(mAnalysisMode) {
		case kAnalyzeNone:
			break;
		case kAnalyzeColors:
			for(int i=0; i<9; ++i)
				mpDst[i*2+0] = mpDst[i*2+1] = mColorTable[i];
			break;
		case kAnalyzeDList:
			mpDst[0] = mpDst[1] = (dlControl & 0x80) ? 0x1f : 0x00;
			mpDst[2] = mpDst[3] = (dlControl & 0x40) ? 0x3f : 0x00;
			mpDst[4] = mpDst[5] = (dlControl & 0x20) ? 0x5f : 0x00;
			mpDst[6] = mpDst[7] = (dlControl & 0x10) ? 0x7f : 0x00;
			mpDst[8] = mpDst[9] = mpDst[10] = mpDst[11] = ((dlControl & 0x0f) << 4) + 15;
			break;
	}
}

void ATGTIAEmulator::UpdatePlayer(int index, uint8 byte) {
	if (mGRACTL & 2) {
		mDelayedPlayerData[index] = byte;
		if (!(mVDELAY & (0x10 << index)))
			mPlayerData[index] = byte;
	}
}

void ATGTIAEmulator::UpdateMissile(uint8 byte) {
	if (mGRACTL & 1) {
		mDelayedMissileData = byte;

		uint8 mask = 0;
		if (mVDELAY & 1)
			mask |= 3;
		if (mVDELAY & 2)
			mask |= 0x0c;
		if (mVDELAY & 4)
			mask |= 0x30;
		if (mVDELAY & 8)
			mask |= 0xc0;

		mMissileData ^= (mMissileData ^ byte) & ~mask;
	}
}

void ATGTIAEmulator::UpdatePlayfield160(uint32 x, uint8 byte) {
	uint8 *dst = &mMergeBuffer[x*2];

	dst[0] = (byte >>  4) & 15;
	dst[1] = (byte      ) & 15;
}

void ATGTIAEmulator::UpdatePlayfield320(uint32 x, uint8 byte) {
	uint8 *dst = &mMergeBuffer[x*2];
	dst[0] = dst[1] = PF2;

	int ix = x >> 1;
	if (x & 1)
		mPlayfieldLumaBytes[ix] += byte;
	else
		mPlayfieldLumaBytes[ix] = byte << 4;
}

void ATGTIAEmulator::EndPlayfield() {
}

namespace {
	uint32 Expand(uint8 x, uint8 mode) {
		static const uint8 tab2[16]={
			0x00,
			0x03,
			0x0c,
			0x0f,
			0x30,
			0x33,
			0x3c,
			0x3f,
			0xc0,
			0xc3,
			0xcc,
			0xcf,
			0xf0,
			0xf3,
			0xfc,
			0xff,
		};
		static const uint16 tab4[16]={
			0x0000,
			0x000f,
			0x00f0,
			0x00ff,
			0x0f00,
			0x0f0f,
			0x0ff0,
			0x0fff,
			0xf000,
			0xf00f,
			0xf0f0,
			0xf0ff,
			0xff00,
			0xff0f,
			0xfff0,
			0xffff,
		};

		switch(mode) {
			default:
				return (uint32)x << 24;
			case 1:
				return ((uint32)tab2[x >> 4] << 24) + ((uint32)tab2[x & 15] << 16);
			case 3:
				return ((uint32)tab4[x >> 4] << 16) + (uint32)tab4[x & 15];
		}
	}
}

void ATGTIAEmulator::Sync() {
	mpConn->GTIARequestAnticSync();

	uint32 targetX = mpConn->GTIAGetXClock();
	int x1 = mLastSyncX * 2;
	int x2 = targetX*2;
	mLastSyncX = targetX;

	if (x1 < 34)
		x1 = 34;
	if (x2 > 222)
		x2 = 222;
	if (x1 >= x2)
		return;

	// obey VBLANK
	if (mY < 8 || mY >= 248)
		return;

	uint8 *dst = mMergeBuffer;

	for(uint32 player=0; player<4; ++player) {
		uint8 data = mPlayerData[player];

		if (data) {
			static const int kPlayerWidths[4]={8,16,8,32};

			int px = mPlayerPos[player];
			int px1 = px;
			int px2 = px + kPlayerWidths[mPlayerSize[player]];

			if (px1 < x1)
				px1 = x1;
			if (px2 > x2)
				px2 = x2;

			if (px1 != px2) {
				uint8 *pldst = mMergeBuffer + px1;
				uint8 bit = P0 << player;
				sint32 mask = Expand(data, mPlayerSize[player]) << (px1 - px);
				sint32 mask2 = mask;
				uint8 flags = 0;
				for(int x=px2-px1; x >= 0; --x) {
					if (mask < 0) {
						flags |= *pldst;
						*pldst |= bit;
					}

					++pldst;
					mask += mask;
				}

				if (mbHiresMode && !(mPRIOR & 0xc0)) {
					flags &= ~PF;

					for(int x=px1; x < px2; ++x) {
						if (mask2 < 0) {
							if (mPlayfieldLumaBytes[x >> 2] & (0xc0 >> ((x & 3)*2)))
								flags |= PF2;
						}

						++pldst;
						mask2 += mask2;
					}
				}

				if (flags)
					mPlayerCollFlags[player] |= flags;
			}
		}
	}

	static const int kMissileShifts[4]={6,4,2,0};
	static const int kMissileWidths[4]={2,4,2,8};

	if (mMissileData) {
		for(uint32 missile=0; missile<4; ++missile) {
			uint8 data = (mMissileData << kMissileShifts[missile]) & 0xc0;

			if (data) {
				int px = mMissilePos[missile];
				int px1 = px;
				int px2 = px + kMissileWidths[(mMissileSize >> (2*missile)) & 3];

				if (px1 < x1)
					px1 = x1;
				if (px2 > x2)
					px2 = x2;

				if (px1 != px2) {
					uint8 *pldst = mMergeBuffer + px1;
					sint32 mask = Expand(data, (mMissileSize >> (2*missile)) & 3) << (px1 - px);
					sint32 mask2 = mask;
					uint8 flags = 0;
					for(int x=px2-px1; x >= 0; --x) {
						if (mask < 0)
							flags |= *pldst;

						++pldst;
						mask += mask;
					}

					if (mbHiresMode && !(mPRIOR & 0xc0)) {
						flags &= ~PF;

						for(int x=px1; x < px2; ++x) {
							if (mask2 < 0) {
								if (mPlayfieldLumaBytes[x >> 2] & (0xc0 >> ((x & 3)*2)))
									flags |= PF2;
							}

							++pldst;
							mask2 += mask2;
						}
					}

					if (flags)
						mMissileCollFlags[missile] |= flags;
				}
			}
		}

		for(uint32 missile=0; missile<4; ++missile) {
			uint8 data = (mMissileData << kMissileShifts[missile]) & 0xc0;

			if (data) {
				int px = mMissilePos[missile];
				int px1 = px;
				int px2 = px + kMissileWidths[(mMissileSize >> (2*missile)) & 3];

				if (px1 < x1)
					px1 = x1;
				if (px2 > x2)
					px2 = x2;

				if (px1 != px2) {
					uint8 *pldst = mMergeBuffer + px1;
					uint8 bit = (mPRIOR & 0x10) ? PF3 : P0 << missile;
					sint32 mask = Expand(data, (mMissileSize >> (2*missile)) & 3) << (px1 - px);
					for(int x=px2-px1; x >= 0; --x) {
						if (mask < 0)
							*pldst |= bit;

						++pldst;
						mask += mask;
					}
				}
			}
		}
	}

	// render scanline
	if (!mpDst)
		return;

	dst = mpDst + x1*2;
	const uint8 *src = mMergeBuffer + x1;

	static const uint32 kHiresMask = (1<<2) | (1<<3) | (1<<15);

	int w = x2 - x1;

	static const uint8 kAnalysisColorTable[]={
		0x1a,
		0x5a,
		0x7a,
		0x9a,
		0x03,
		0x07,
		0x0b,
		0x0f,
		0x00,
		0x00,
		0x00,
		0x00,
		0x03,
		0x07,
		0x0b,
		0x0f,
		0xe4,
		0x3a,
		0x8a
	};

	const uint8 *__restrict colorTable = mAnalysisMode == kAnalyzeLayers ? kAnalysisColorTable : mColorTable;
	const uint8 *__restrict priTable = mpPriTable;

	if (x1 == 34)
		memset(mpDst, colorTable[8], 68);

	if (mbHiresMode) {
		uint8 luma1 = mPFColor[1] & 0xf;
		const uint8 *lumasrc = &mPlayfieldLumaBytes[x1 >> 2];

		bool oddfirst = (x1 & 2) != 0;
		bool oddlast = false;
		int pairs = 0;

		if (x1+1 < x2) {
			oddlast = (x2 & 2) != 0;
			pairs = (x2 - ((x1 + 3) & ~3)) >> 2;
		}

		uint8 bak = mPFBAK;

		switch(mPRIOR & 0xc0) {
			case 0x00:
				if (oddfirst) {
					uint8 c4 = colorTable[priTable[*src++]];
					uint8 c5 = c4;
					uint8 c6 = colorTable[priTable[*src++]];
					uint8 c7 = c6;

					uint8 lb = *lumasrc++;

					if (lb & 0x08) c4 = (c4 & 0xf0) + luma1;
					if (lb & 0x04) c5 = (c5 & 0xf0) + luma1;
					if (lb & 0x02) c6 = (c6 & 0xf0) + luma1;
					if (lb & 0x01) c7 = (c7 & 0xf0) + luma1;

					dst[0] = c4;
					dst[1] = c5;
					dst[2] = c6;
					dst[3] = c7;
					dst += 4;
				}

				while(pairs--) {
					uint8 c0 = colorTable[priTable[*src++]];
					uint8 c1 = c0;
					uint8 c2 = colorTable[priTable[*src++]];
					uint8 c3 = c2;
					uint8 c4 = colorTable[priTable[*src++]];
					uint8 c5 = c4;
					uint8 c6 = colorTable[priTable[*src++]];
					uint8 c7 = c6;

					uint8 lb = *lumasrc++;

					if (lb & 0x80) c0 = (c0 & 0xf0) + luma1;
					if (lb & 0x40) c1 = (c1 & 0xf0) + luma1;
					if (lb & 0x20) c2 = (c2 & 0xf0) + luma1;
					if (lb & 0x10) c3 = (c3 & 0xf0) + luma1;
					if (lb & 0x08) c4 = (c4 & 0xf0) + luma1;
					if (lb & 0x04) c5 = (c5 & 0xf0) + luma1;
					if (lb & 0x02) c6 = (c6 & 0xf0) + luma1;
					if (lb & 0x01) c7 = (c7 & 0xf0) + luma1;

					dst[0] = c0;
					dst[1] = c1;
					dst[2] = c2;
					dst[3] = c3;
					dst[4] = c4;
					dst[5] = c5;
					dst[6] = c6;
					dst[7] = c7;
					dst += 8;
				}

				if (oddlast) {
					uint8 c0 = colorTable[priTable[*src++]];
					uint8 c1 = c0;
					uint8 c2 = colorTable[priTable[*src++]];
					uint8 c3 = c2;

					uint8 lb = *lumasrc;

					if (lb & 0x80) c0 = (c0 & 0xf0) + luma1;
					if (lb & 0x40) c1 = (c1 & 0xf0) + luma1;
					if (lb & 0x20) c2 = (c2 & 0xf0) + luma1;
					if (lb & 0x10) c3 = (c3 & 0xf0) + luma1;

					dst[0] = c0;
					dst[1] = c1;
					dst[2] = c2;
					dst[3] = c3;
				}
				break;

			case 0x40:	// 1 color / 16 lumas
				if (oddfirst) {
					uint8 pri2 = priTable[*src++];
					uint8 pri3 = priTable[*src++];

					uint8 lb = *lumasrc++;
					uint8 l1 = lb & 15;
					uint8 gc1 = bak | l1;

					uint8 c4 = (uint8)(pri2-4) < 5 ? gc1 : colorTable[pri2];
					uint8 c6 = (uint8)(pri3-4) < 5 ? gc1 : colorTable[pri3];

					dst[0] = dst[1] = c4;
					dst[2] = dst[3] = c6;
					dst += 4;
				}

				while(pairs--) {
					uint8 pri0 = priTable[*src++];
					uint8 pri1 = priTable[*src++];
					uint8 pri2 = priTable[*src++];
					uint8 pri3 = priTable[*src++];

					uint8 lb = *lumasrc++;
					uint8 l0 = lb >> 4;
					uint8 l1 = lb & 15;
					uint8 gc0 = bak | l0;
					uint8 gc1 = bak | l1;

					uint8 c0 = (uint8)(pri0-4) < 5 ? gc0 : colorTable[pri0];
					uint8 c2 = (uint8)(pri1-4) < 5 ? gc0 : colorTable[pri1];
					uint8 c4 = (uint8)(pri2-4) < 5 ? gc1 : colorTable[pri2];
					uint8 c6 = (uint8)(pri3-4) < 5 ? gc1 : colorTable[pri3];

					dst[0] = dst[1] = c0;
					dst[2] = dst[3] = c2;
					dst[4] = dst[5] = c4;
					dst[6] = dst[7] = c6;
					dst += 8;
				}

				if (oddlast) {
					uint8 pri0 = priTable[*src++];
					uint8 pri1 = priTable[*src++];

					uint8 lb = *lumasrc++;
					uint8 l0 = lb >> 4;
					uint8 gc0 = bak | l0;

					uint8 c0 = (uint8)(pri0-4) < 5 ? gc0 : colorTable[pri0];
					uint8 c2 = (uint8)(pri1-4) < 5 ? gc0 : colorTable[pri1];

					dst[0] = dst[1] = c0;
					dst[2] = dst[3] = c2;
				}
				break;

			case 0x80:	// 9 colors
				if (oddfirst) {
					uint8 lb = *lumasrc++;
					uint8 l1 = lb & 15;

					uint8 c4 = colorTable[l1];

					dst[0] = dst[1] = c4;
					dst[2] = dst[3] = c4;
					dst += 4;
				}

				while(pairs--) {
					uint8 lb = *lumasrc++;
					uint8 l0 = lb >> 4;
					uint8 l1 = lb & 15;

					uint8 c0 = colorTable[l0];
					uint8 c4 = colorTable[l1];

					dst[0] = dst[1] = c0;
					dst[2] = dst[3] = c0;
					dst[4] = dst[5] = c4;
					dst[6] = dst[7] = c4;
					dst += 8;
				}

				if (oddlast) {
					uint8 lb = *lumasrc++;
					uint8 l0 = lb >> 4;

					uint8 c0 = colorTable[l0];

					dst[0] = dst[1] = c0;
					dst[2] = dst[3] = c0;
					dst += 4;
				}
				break;

			case 0xC0:	// 16 colors / 1 luma
				if (oddfirst) {
					uint8 pri2 = priTable[*src++];
					uint8 pri3 = priTable[*src++];

					uint8 lb = *lumasrc++;
					uint8 l1 = lb << 4;

					uint8 gc1 = bak | l1;

					uint8 c4 = (uint8)(pri2 - 4) < 5 ? gc1 : colorTable[pri2];
					uint8 c6 = (uint8)(pri3 - 4) < 5 ? gc1 : colorTable[pri3];

					dst[0] = dst[1] = c4;
					dst[2] = dst[3] = c6;
					dst += 4;
				}

				while(pairs--) {
					uint8 pri0 = priTable[*src++];
					uint8 pri1 = priTable[*src++];
					uint8 pri2 = priTable[*src++];
					uint8 pri3 = priTable[*src++];

					uint8 lb = *lumasrc++;
					uint8 l0 = lb & 0xf0;
					uint8 l1 = lb << 4;

					uint8 gc0 = bak | l0;
					uint8 gc1 = bak | l1;

					uint8 c0 = (uint8)(pri0 - 4) < 5 ? gc0 : colorTable[pri0];
					uint8 c2 = (uint8)(pri1 - 4) < 5 ? gc0 : colorTable[pri1];
					uint8 c4 = (uint8)(pri2 - 4) < 5 ? gc1 : colorTable[pri2];
					uint8 c6 = (uint8)(pri3 - 4) < 5 ? gc1 : colorTable[pri3];

					dst[0] = dst[1] = c0;
					dst[2] = dst[3] = c2;
					dst[4] = dst[5] = c4;
					dst[6] = dst[7] = c6;
					dst += 8;
				}

				if (oddlast) {
					uint8 pri0 = priTable[*src++];
					uint8 pri1 = priTable[*src++];

					uint8 lb = *lumasrc++;
					uint8 l0 = lb & 0xf0;

					uint8 gc0 = bak | l0;

					uint8 c0 = (uint8)(pri0 - 4) < 5 ? gc0 : colorTable[pri0];
					uint8 c2 = (uint8)(pri1 - 4) < 5 ? gc0 : colorTable[pri1];

					dst[0] = dst[1] = c0;
					dst[2] = dst[3] = c2;
				}
				break;
		}
	} else {
		int w4 = w >> 2;

		for(int i=0; i<w4; ++i) {
			dst[0] = dst[1] = colorTable[priTable[src[0]]];
			dst[2] = dst[3] = colorTable[priTable[src[1]]];
			dst[4] = dst[5] = colorTable[priTable[src[2]]];
			dst[6] = dst[7] = colorTable[priTable[src[3]]];
			src += 4;
			dst += 8;
		}

		for(int i=w & 3; i; --i) {
			dst[0] = dst[1] = colorTable[priTable[src[0]]];
			++src;
			dst += 2;
		}
	}

	if (x2 == 222)
		memset(mpDst + 444, colorTable[8], 456 - 444);
}

void ATGTIAEmulator::RenderActivityMap(const uint8 *src) {
	if (!mpFrame)
		return;

	uint8 *dst = (uint8 *)mpFrame->mPixmap.data;
//	src += 16;
	for(int y=0; y<262; ++y) {
		uint8 *dst2 = dst;

//		for(int x=0; x<96; ++x) {
		for(int x=0; x<114; ++x) {
			uint8 add = src[x] & 1 ? 0x08 : 0x00;
			dst2[0] = (dst2[0] & 0xf0) + ((dst2[0] & 0xf) >> 1) + add;
			dst2[1] = (dst2[1] & 0xf0) + ((dst2[1] & 0xf) >> 1) + add;
			dst2[2] = (dst2[2] & 0xf0) + ((dst2[2] & 0xf) >> 1) + add;
			dst2[3] = (dst2[3] & 0xf0) + ((dst2[3] & 0xf) >> 1) + add;
			dst2 += 4;
		}
		src += 114;
		dst += mpFrame->mPixmap.pitch;
	}
}

void ATGTIAEmulator::UpdateScreen() {
	if (!mpFrame)
		return;

	const VDPixmap& pxdst = mArtifactMode ? mPreArtifactFrame : mpFrame->mPixmap;

	if (mY < 261) {
		Sync();
		uint32 x = mpConn->GTIAGetXClock();

		uint8 *row = (uint8 *)pxdst.data + (mY+1)*pxdst.pitch;
		VDMemset8(row, 0x00, 4*x);
		VDMemset8(row + x*4, 0xFF, 456 - 4*x);

		ApplyArtifacting();

		mpDisplay->SetSourcePersistent(true, mpFrame->mPixmap);
	} else {
		uint32 statusFlags = mStatusFlags | mStickyStatusFlags;
		mStickyStatusFlags = mStatusFlags;

		static const uint8 chars[4][7][5]={
			{
#define X 0x1F
				X,X,X,X,X,
				X,X,0,X,X,
				X,0,0,X,X,
				X,X,0,X,X,
				X,X,0,X,X,
				X,0,0,0,X,
				X,X,X,X,X,
#undef X
			},
			{
#define X 0x3F
				X,X,X,X,X,
				X,0,0,X,X,
				X,X,X,0,X,
				X,X,0,X,X,
				X,0,X,X,X,
				X,0,0,0,X,
				X,X,X,X,X,
#undef X
			},
			{
#define X 0x5F
				X,X,X,X,X,
				X,0,0,X,X,
				X,X,X,0,X,
				X,X,0,X,X,
				X,X,X,0,X,
				X,0,0,X,X,
				X,X,X,X,X,
#undef X
			},
			{
#define X 0x7F
				X,X,X,X,X,
				X,0,X,0,X,
				X,0,X,0,X,
				X,0,0,0,X,
				X,X,X,0,X,
				X,X,X,0,X,
				X,X,X,X,X,
#undef X
			},
		};

		for(int i=0; i<4; ++i) {
			if (statusFlags & (1 << i)) {
				VDPixmap pxsrc;
				pxsrc.data = (void *)chars[i];
				pxsrc.pitch = sizeof chars[i][0];
				pxsrc.format = nsVDPixmap::kPixFormat_Pal8;
				pxsrc.palette = mPalette;
				pxsrc.w = 5;
				pxsrc.h = 7;

				VDPixmapBlt(mpFrame->mPixmap, mpFrame->mPixmap.w - 20 + 5*i, mpFrame->mPixmap.h - 7, pxsrc, 0, 0, 5, 7);
			}
		}

		if (mbShowCassetteIndicator) {
			static const int kIndices[]={
				0,1,2,0,2,3,
				4,5,6,4,6,7,
				8,9,10,8,10,11,
				12,13,14,12,14,15,
				16,17,18,16,18,19,
			};

			VDTriColorVertex vx[20];

			float f = (mCassettePos - floor(mCassettePos)) * 20.0f;

			for(int i=0; i<20; i+=4) {
				float a1 = (float)(i + f) * nsVDMath::kfTwoPi / 20.0f;
				float a2 = (float)(i + f + 2.0f) * nsVDMath::kfTwoPi / 20.0f;
				float x1 = cosf(a1);
				float y1 = -sinf(a1);
				float x2 = cosf(a2);
				float y2 = -sinf(a2);

				vx[i+0].x = 424.0f + 10.0f*x1;
				vx[i+0].y = 222.0f + 10.0f*y1;
				vx[i+0].z = 0.0f;
				vx[i+0].a = 255;
				vx[i+0].r = 255;
				vx[i+0].g = 0x0E;
				vx[i+0].b = 255;

				vx[i+1].x = 424.0f + 15.0f*x1;
				vx[i+1].y = 222.0f + 15.0f*y1;
				vx[i+1].z = 0.0f;
				vx[i+1].a = 255;
				vx[i+1].r = 255;
				vx[i+1].g = 0x0E;
				vx[i+1].b = 255;

				vx[i+2].x = 424.0f + 15.0f*x2;
				vx[i+2].y = 222.0f + 15.0f*y2;
				vx[i+2].z = 0.0f;
				vx[i+2].a = 255;
				vx[i+2].r = 255;
				vx[i+2].g = 0x0E;
				vx[i+2].b = 255;

				vx[i+3].x = 424.0f + 10.0f*x2;
				vx[i+3].y = 222.0f + 10.0f*y2;
				vx[i+3].z = 0.0f;
				vx[i+3].a = 255;
				vx[i+3].r = 255;
				vx[i+3].g = 0x0E;
				vx[i+3].b = 255;
			}

			const float xf[16]={
				2.0f / 456.0f, 0, 0, -1.0f,
				0, -2.0f / 262.0f, 0, 1.0f,
				0, 0, 0, 0,
				0, 0, 0, 1.0f
			};

			VDPixmap tmp(mpFrame->mPixmap);
			tmp.format = nsVDPixmap::kPixFormat_Y8;
			VDPixmapTriFill(tmp, vx, 20, kIndices, 30, xf);
		}

		ApplyArtifacting();
		mpDisplay->PostBuffer(mpFrame);
		mpFrame = NULL;
	}
}

void ATGTIAEmulator::RecomputePalette() {
	uint32 *dst = mPalette;
	for(int luma = 0; luma<16; ++luma) {
		*dst++ = 0x111111 * luma;
	}

#if 0
	for(int hue=0; hue<15; ++hue) {
		double angle = (double)(-hue + 12) * nsVDMath::krTwoPi * (1.0 / 15.0);
		double r0 = 0.5*cos(angle + nsVDMath::krTwoPi * 1.0 / 3.0);
		double g0 = 0.5*cos(angle);
		double b0 = 0.5*cos(angle + nsVDMath::krTwoPi * 2.0 / 3.0);

		for(int luma=0; luma<16; ++luma) {
			double y = (double)luma / 15.0 * 1.3 - 0.2;
			double r = y + r0;
			double g = y + g0;
			double b = y + b0;

			*dst++	= (VDClampedRoundFixedToUint8Fast((float)r) << 16)
					+ (VDClampedRoundFixedToUint8Fast((float)g) <<  8)
					+ (VDClampedRoundFixedToUint8Fast((float)b)      );
		}
	}
#elif 0
	for(int hue=0; hue<15; ++hue) {
		double angle = (double)(-hue + 11.2) * nsVDMath::krTwoPi * (1.0 / 15.0);
		double r0 = 0.5*cos(angle + nsVDMath::krTwoPi * 1.0 / 3.0) * 0.8;
		double g0 = 0.5*cos(angle) * 0.8;
		double b0 = 0.5*cos(angle + nsVDMath::krTwoPi * 2.0 / 3.0) * 0.8;
		double y0 = 0.30*r0 + 0.59*g0 + 0.11*b0;

		r0 -= y0;
		g0 -= y0;
		b0 -= y0;

		for(int luma=0; luma<16; ++luma) {
//			double y = (double)luma / 15.0 * 1.3 - 0.2;
			double y = (double)luma / 15.0;
			double c = sin(luma / 15.0 * nsVDMath::krPi) * 0.5 + 0.5;
			double r = y + r0 * c;
			double g = y + g0 * c;
			double b = y + b0 * c;

#if 0
			if (r > 1.0 || g > 1.0 || b > 1.0) {
				double y2 = 0.30*r + 0.59*g + 0.11*b;
				double mx = r > g ? r > b ? r : b : g > b ? g : b;

				r = (r - y2) * (1.0 - y2) / (mx - y2) + y2;
				g = (g - y2) * (1.0 - y2) / (mx - y2) + y2;
				b = (b - y2) * (1.0 - y2) / (mx - y2) + y2;
			}
#endif

			*dst++	= (VDClampedRoundFixedToUint8Fast((float)r) << 16)
					+ (VDClampedRoundFixedToUint8Fast((float)g) <<  8)
					+ (VDClampedRoundFixedToUint8Fast((float)b)      );
		}
	}
#else
	double angle = mPaletteColorBase * nsVDMath::krTwoPi;
	double angleStep = nsVDMath::krTwoPi / mPaletteColorRange;

	for(int hue=0; hue<15; ++hue) {
#if 0
		double i = (50.0/255.0) * cos(angle);
		double q = (50.0/255.0) * sin(angle);
#else
		double i = (75.0/255.0) * cos(angle);
		double q = (75.0/255.0) * sin(angle);
#endif

		for(int luma=0; luma<16; ++luma) {
			double y = (double)(luma + 1) / 16.0;
			double r = y + 0.956*i + 0.621*q;
			double g = y - 0.272*i - 0.647*q;
			double b = y - 1.107*i + 1.704*q;

			*dst++	= (VDClampedRoundFixedToUint8Fast((float)r) << 16)
					+ (VDClampedRoundFixedToUint8Fast((float)g) <<  8)
					+ (VDClampedRoundFixedToUint8Fast((float)b)      );
		}

		angle += angleStep;
	}
#endif
}

uint8 ATGTIAEmulator::ReadByte(uint8 reg) {
	// fast registers
	switch(reg) {
		case 0x10:
		case 0x11:
		case 0x12:
		case 0x13:
			return mTRIG[reg - 0x10];
		case 0x14:
			return mbPALMode ? 0x00 : 0x0E;
		case 0x15:
		case 0x16:
		case 0x17:	// must return LSB0 set or Recycle hangs
		case 0x18:
		case 0x19:
		case 0x1A:
		case 0x1B:
		case 0x1C:
		case 0x1D:
		case 0x1E:
			return 0x0F;
		case 0x1F:		// $D01F CONSOL
			return mCONSOL & 0x07;
	}

	Sync();	

	switch(reg) {
		// missile-to-playfield collisions
		case 0x00:	return mMissileCollFlags[0] & 15;
		case 0x01:	return mMissileCollFlags[1] & 15;
		case 0x02:	return mMissileCollFlags[2] & 15;
		case 0x03:	return mMissileCollFlags[3] & 15;

		// player-to-playfield collisions
		case 0x04:	return mPlayerCollFlags[0] & 15;
		case 0x05:	return mPlayerCollFlags[1] & 15;
		case 0x06:	return mPlayerCollFlags[2] & 15;
		case 0x07:	return mPlayerCollFlags[3] & 15;

		// missile-to-player collisions
		case 0x08:	return mMissileCollFlags[0] >> 4;
		case 0x09:	return mMissileCollFlags[1] >> 4;
		case 0x0A:	return mMissileCollFlags[2] >> 4;
		case 0x0B:	return mMissileCollFlags[3] >> 4;

		// player-to-player collisions
		case 0x0C:	return    ((mPlayerCollFlags[1] >> 3) & 0x02)	// 1 -> 0
							+ ((mPlayerCollFlags[2] >> 2) & 0x04)	// 2 -> 0
							+ ((mPlayerCollFlags[3] >> 1) & 0x08);	// 3 -> 0

		case 0x0D:	return    ((mPlayerCollFlags[1] >> 4) & 0x01)	// 1 -> 0
							+ ((mPlayerCollFlags[2] >> 3) & 0x04)	// 2 -> 1
							+ ((mPlayerCollFlags[3] >> 2) & 0x08);	// 3 -> 1

		case 0x0E:	return    ((mPlayerCollFlags[2] >> 4) & 0x03)	// 2 -> 0, 1
							+ ((mPlayerCollFlags[3] >> 3) & 0x08);	// 3 -> 2

		case 0x0F:	return    ((mPlayerCollFlags[3] >> 4) & 0x07);	// 3 -> 0, 1, 2

		default:
//			__debugbreak();
			break;
	}
	return 0;
}

void ATGTIAEmulator::WriteByte(uint8 reg, uint8 value) {
	switch(reg) {
		case 0x1C:
			mVDELAY = value;
			break;

		case 0x1D:
			mGRACTL = value;
			break;

		case 0x1F:		// $D01F CONSOL
			{
				uint8 newConsol = (mCONSOL & 0xf7) + (value & 0x08);
				if ((newConsol ^ mCONSOL) & 8)
					mpConn->GTIASetSpeaker(0 != (newConsol & 8));
				mCONSOL = newConsol;
			}
			break;
	}

	Sync();

	switch(reg) {
		case 0x00:	mPlayerPos[0] = value;			break;
		case 0x01:	mPlayerPos[1] = value;			break;
		case 0x02:	mPlayerPos[2] = value;			break;
		case 0x03:	mPlayerPos[3] = value;			break;
		case 0x04:	mMissilePos[0] = value;			break;
		case 0x05:	mMissilePos[1] = value;			break;
		case 0x06:	mMissilePos[2] = value;			break;
		case 0x07:	mMissilePos[3] = value;			break;
		case 0x08:	mPlayerSize[0] = value & 3;			break;
		case 0x09:	mPlayerSize[1] = value & 3;			break;
		case 0x0A:	mPlayerSize[2] = value & 3;			break;
		case 0x0B:	mPlayerSize[3] = value & 3;			break;
		case 0x0C:	mMissileSize = value;			break;
		case 0x0D:
			mDelayedPlayerData[0] = value;
			if (!(mVDELAY & 0x10))
				mPlayerData[0] = value;
			break;
		case 0x0E:
			mDelayedPlayerData[1] = value;
			if (!(mVDELAY & 0x20))
				mPlayerData[1] = value;
			break;
		case 0x0F:
			mDelayedPlayerData[2] = value;
			if (!(mVDELAY & 0x40))
				mPlayerData[2] = value;
			break;
		case 0x10:
			mDelayedPlayerData[3] = value;
			if (!(mVDELAY & 0x80))
				mPlayerData[3] = value;
			break;
		case 0x11:
			mDelayedMissileData = value;
			{
				uint8 mask = 0;
				if (mVDELAY & 1)
					mask |= 3;
				if (mVDELAY & 2)
					mask |= 0x0c;
				if (mVDELAY & 4)
					mask |= 0x30;
				if (mVDELAY & 8)
					mask |= 0xc0;

				mMissileData ^= (mMissileData ^ value) & ~mask;
			}
			break;
		case 0x12:
			mPMColor[0] = value;
			mColorTable[0] = value;
			mColorTable[17] = mPMColor[0] | mPMColor[1];
			break;
		case 0x13:	mPMColor[1] = value;
			mColorTable[1] = value;
			mColorTable[17] = mPMColor[0] | mPMColor[1];
			break;
		case 0x14:	mPMColor[2] = value;
			mColorTable[2] = value;
			mColorTable[18] = mPMColor[2] | mPMColor[3];
			break;
		case 0x15:	mPMColor[3] = value;
			mColorTable[3] = value;
			mColorTable[18] = mPMColor[2] | mPMColor[3];
			break;
		case 0x16:	mPFColor[0] = value;
			mColorTable[4] = value;
			mColorTable[12] = value;
			break;
		case 0x17:	mPFColor[1] = value;
			mColorTable[5] = value;
			mColorTable[13] = value;
			break;
		case 0x18:	mPFColor[2] = value;
			mColorTable[6] = value;
			mColorTable[14] = value;
			break;
		case 0x19:	mPFColor[3] = value;
			mColorTable[7] = value;
			mColorTable[15] = value;
			break;
		case 0x1A:	mPFBAK = value;
			mColorTable[8] = value;
			mColorTable[9] = value;
			mColorTable[10] = value;
			mColorTable[11] = value;
			break;
		case 0x1B:
			mPRIOR = value;
			mpPriTable = mPriorityTables[(value & 15) + (value&32 ? 16 : 0)];
//			mpPriTable = mPriorityTables[1];
			if (mPRIOR & 16)
				mpMissileTable = kMissileTables[1];
			else
				mpMissileTable = kMissileTables[0];
			break;
		case 0x1E:		// $D01E HITCLR
			memset(mPlayerCollFlags, 0, sizeof mPlayerCollFlags);
			memset(mMissileCollFlags, 0, sizeof mMissileCollFlags);
			break;
		default:
//			__debugbreak();
			break;
	}
}

void ATGTIAEmulator::ApplyArtifacting() {
	if (!mArtifactMode)
		return;

	float yr[912+14];

	char *dstrow = (char *)mpFrame->mPixmap.data;
	ptrdiff_t dstpitch = mpFrame->mPixmap.pitch;

	const uint8 *srcrow = (const uint8 *)mPreArtifactFrame.data;
	ptrdiff_t srcpitch = mPreArtifactFrame.pitch;

	for(uint32 row=0; row<262; ++row) {
		uint32 *dst = (uint32 *)dstrow;
		const uint8 *src = srcrow;

		float *ydst = yr+7;
		for(uint32 x=0; x<456; ++x) {
			uint8 c = *src++;

			float i = 0;
			float q = 0;
			float y1 = (float)(c & 15) / 15.0f;
			float y2 = y1;

			if (c & 0xf0) {
				float phase0 = (float)((c & 0xf0) - 0x10) * (3.1415926535f * 2.0f / 15.0f) + (float)(x & 1) * 3.1415926535f;

				y1 += cosf(phase0) * 0.5f;
				y2 -= sinf(phase0) * 0.5f;
			}

			ydst[0] = y1;
			ydst[1] = y2;
			ydst += 2;
		}

		for(int x=0; x<7; ++x) {
			yr[x] = yr[7];
			yr[912+7+x] = yr[912+6];
		}

		ydst = yr;
		for(int x=0; x<912; ++x) {
//			float y = ydst[5]*(1.0f/16.0f) + ydst[6]*(4.0f/16.0f) + ydst[7]*(6.0f/16.0f) + ydst[8]*(4.0f/16.0f) + ydst[9]*(1.0f/16.0f);
			float y = ydst[ 2]*(  1.0f/1024.0f)
					+ ydst[ 3]*( 10.0f/1024.0f)
					+ ydst[ 4]*( 45.0f/1024.0f)
					+ ydst[ 5]*(120.0f/1024.0f)
					+ ydst[ 6]*(210.0f/1024.0f)
					+ ydst[ 7]*(252.0f/1024.0f)
					+ ydst[ 8]*(210.0f/1024.0f)
					+ ydst[ 9]*(120.0f/1024.0f)
					+ ydst[10]*( 45.0f/1024.0f)
					+ ydst[11]*( 10.0f/1024.0f)
					+ ydst[12]*(  1.0f/1024.0f)
					;

			float i;
			float q;
			
			switch(x & 3) {
				case 0:
					i	= ydst[5]*(1.0f/4.0f)*-1
						+ ydst[7]*(2.0f/4.0f)*+1
						+ ydst[9]*(1.0f/4.0f)*-1;

					q	= ydst[ 4]*(1.0f/8.0f)*+1
						+ ydst[ 6]*(3.0f/8.0f)*-1
						+ ydst[ 8]*(3.0f/8.0f)*+1
						+ ydst[10]*(1.0f/8.0f)*-1
						;
					break;

				case 1:
					i	= ydst[ 4]*(1.0f/8.0f)*-1
						+ ydst[ 6]*(3.0f/8.0f)*+1
						+ ydst[ 8]*(3.0f/8.0f)*-1
						+ ydst[10]*(1.0f/8.0f)*+1
						;

					q	= ydst[5]*(1.0f/4.0f)*-1
						+ ydst[7]*(2.0f/4.0f)*+1
						+ ydst[9]*(1.0f/4.0f)*-1
						;
					break;

				case 2:
					i	= ydst[5]*(1.0f/4.0f)*+1
						+ ydst[7]*(2.0f/4.0f)*-1
						+ ydst[9]*(1.0f/4.0f)*+1
						;

					q	= ydst[ 4]*(1.0f/8.0f)*-1
						+ ydst[ 6]*(3.0f/8.0f)*+1
						+ ydst[ 8]*(3.0f/8.0f)*-1
						+ ydst[10]*(1.0f/8.0f)*+1
						;
					break;

				case 3:
					i	= ydst[ 4]*(1.0f/8.0f)*+1
						+ ydst[ 6]*(3.0f/8.0f)*-1
						+ ydst[ 8]*(3.0f/8.0f)*+1
						+ ydst[10]*(1.0f/8.0f)*-1
						;

					q	= ydst[5]*(1.0f/4.0f)*+1
						+ ydst[7]*(2.0f/4.0f)*-1
						+ ydst[9]*(1.0f/4.0f)*+1
						;
					break;
			}
q=-q;
			++ydst;

			float red = y + 0.9563f*i + 0.6210f*q;
			float grn = y - 0.2721f*i - 0.6474f*q;
			float blu = y - 1.1070f*i + 1.7046f*q;

			uint8 ir = VDClampedRoundFixedToUint8Fast(red);
			uint8 ig = VDClampedRoundFixedToUint8Fast(grn);
			uint8 ib = VDClampedRoundFixedToUint8Fast(blu);

			*dst++ = ((uint32)ir << 16) + ((uint32)ig << 8) + ib;
		}

		srcrow += srcpitch;
		dstrow += dstpitch;
	}
}
